//-------------------------------------------------------//
// File:  Segmenter.cpp                                  //
//                                                       //
// Description: Segmenter Source File                    //
//                                                       //
// Authors: Anthousis Andreadis - http://anthousis.com   //
// Date: 7-Oct-2015                                      //
//                                                       //
// Computer Graphics Group                               //
// http://graphics.cs.aueb.gr/graphics/                  //
// AUEB - Athens University of Economics and Business    //
//                                                       //
//                                                       //
// This work was funded by the EU-FP7 - PRESIOUS project //
//-------------------------------------------------------//

#include "Utils.h"
#include "Segmenter.h"

using namespace std;
using namespace vcg;

QTextStream& qStdOut()
{
	static QTextStream ts(stdout);
	return ts;
}

void Segmenter::setMesh(CMeshO *meshObj) {
    meshObj_ = meshObj;
    Segmenter::faceClusterPairs = tri::Allocator<CMeshO>::GetPerFaceAttribute<Cluster *>(*meshObj_, string("Segment"));
    Segmenter::faceMarks = tri::Allocator<CMeshO>::GetPerFaceAttribute<int>(*meshObj_, string("Mark"));
    Segmenter::faceTmpMarks = tri::Allocator<CMeshO>::GetPerFaceAttribute<int>(*meshObj_, string("TmpMark"));
    Segmenter::faceArea = tri::Allocator<CMeshO>::GetPerFaceAttribute<float>(*meshObj_, string("Area"));
    Segmenter::faceCentroid = tri::Allocator<CMeshO>::GetPerFaceAttribute<vcg::Point3f>(*meshObj_, string("Centroid"));

    for (CMeshO::FaceIterator i = meshObj_->face.begin(); i != meshObj_->face.end(); ++i) {
        Segmenter::faceClusterPairs[*i] = NULL;
        Segmenter::faceMarks[*i] = -1;
        Segmenter::faceTmpMarks[*i] = -1;
        Segmenter::faceArea[*i] = vcg::DoubleArea(*i)*0.5f;
        Segmenter::faceCentroid[*i] = vcg::Barycenter(*i);
    }
    computeTotalAreaAndInitPFA();
}

void Segmenter::postProcessClusters(float areaThreshold) {
    int nClustersBelowThr = 0;
    vector<Cluster *>::iterator s_it, merge_it;
    for (s_it = clusterList_.begin(); s_it != clusterList_.end(); ++s_it) {
        float ratio = 100.f*(*s_it)->getArea() / totalArea_;
        if (ratio < areaThreshold) {
            ++nClustersBelowThr;
        }
    }

    initNumClusters_ = clusterList_.size();
    // Calculate total mesh area
    bool finished;
    do {
        if (cb_) {
            cb_(100 * (initNumClusters_ - clusterList_.size()) / nClustersBelowThr, "Merging segments based on area threshold");
        }
        finished = true;
        Cluster *segToMerge = NULL;
        // Search for a segment that is below the area threshold
        float lowestRatio = FLT_MAX;
        vector<Cluster *>::iterator s_it, merge_it;
        for (s_it = clusterList_.begin(); s_it != clusterList_.end(); ++s_it) {
            float ratio = 100.f*(*s_it)->getArea() / totalArea_;
            if (ratio < areaThreshold && !(*s_it)->disjoined_) {
                if (lowestRatio >= ratio) {
                    lowestRatio = ratio;
                    segToMerge = (*s_it);
                    merge_it = s_it;
                    finished = false;
                }
            }
        }   // end for

        // If we found a valid for merging segment
        if (!finished) {
            decomposeClusterBestFirst(segToMerge);
            // Now clear and remove the merged segment
            merge_it = clusterList_.erase(merge_it);
            delete segToMerge;
            segToMerge = NULL;
        }   // end if
    } while (!finished);    // end do-while
}

void Segmenter::computeTotalAreaAndInitPFA() {
    totalArea_ = 0.f;
    CMeshO::FaceIterator f_it, f_end(meshObj_->face.end());
    for (f_it = meshObj_->face.begin(); f_it != f_end; ++f_it)  {
        faceMarks[*f_it] = -1;
        faceClusterPairs[*f_it] = NULL;

        // If face is deleted ignore it
        if ((*f_it).IsD()) { continue; }
        totalArea_ += Segmenter::faceArea[*f_it];
    }
}

void Segmenter::decomposeClusterBestFirst(Cluster *segToMerge) {
    Cluster *uniqueFNSegIds[100];
    int uIds_pos;
    int *segmentIds = new int[initNumClusters_];
    memset(segmentIds, -1, initNumClusters_ * sizeof(int));
    int round = 0;

    do {
        bool foundFace = false;
        Cluster *bestNeighborSeg = NULL;
        float minWeight = FLT_MAX;
        // Find the best face to merge with a valid neighbor segment
        std::vector<CMeshO::FacePointer>::iterator bestFace;
        size_t bestIndex = 0;
        size_t tmpIndex = 0;
        for (std::vector<CMeshO::FacePointer>::iterator fl_it = segToMerge->faces_.begin(); fl_it != segToMerge->faces_.end(); ++fl_it) {
            CMeshO::FacePointer facePointer = *fl_it;
            if (facePointer->IsD()) { continue; }

            uIds_pos = 0;
            round++;

            // Find unique neighbor segments in 1-ring Face Neighbors
            for (size_t i = 0; i < 3; ++i) {
                CMeshO::FacePointer nFacePointer = facePointer->FFp(i);
                // Sanity checks
                if (!nFacePointer || nFacePointer == facePointer || nFacePointer->IsD()) { continue; }

                int ffSegId = faceMarks[nFacePointer];
                //if (ffSegId != segToMerge->getId() && segmentIds[ffSegId] != round) {
                if (faceClusterPairs[nFacePointer] != segToMerge && segmentIds[ffSegId] != round) {
                    uniqueFNSegIds[uIds_pos] = faceClusterPairs[nFacePointer];
                    uIds_pos++;
                    segmentIds[ffSegId] = round;
                }
            }   // end for

            if (!uIds_pos) {
                ++tmpIndex;
                continue;
            }
            foundFace = true;

            Point3f &faceNormal = facePointer->N();
            //Point3f &faceCentroid = Segmenter::faceCentroid[*facePointer];
            // For each unique segment neighbor of the face search for the best matching one
            for (int i = 0; i < uIds_pos; ++i) {
                Cluster *cn_seg = uniqueFNSegIds[i];
                float weight = metricNormalDeviation(cn_seg->getAvgNormal(), faceNormal);

                if (minWeight >= weight) {
                    minWeight = weight;
                    bestNeighborSeg = cn_seg;
                    bestFace = fl_it;
                    bestIndex = tmpIndex;
                }
                /////////////////////////////////////////////////////////////////
            }   // For that finds the best matching segment for the current Face
            ++tmpIndex;
        }   // For that finds the best face for merging (included the matching segment of it)
        if (!foundFace) {
            segToMerge->disjoined_ = true;
            break;
        }
        clusterAddFace(bestNeighborSeg, (*bestFace));
        // Erase the bestFace from the previous segment
        segToMerge->faces_.erase(bestFace);
    } while (!segToMerge->faces_.empty());
    delete[]segmentIds;
}

bool Segmenter::isEdgeFace(CMeshO::FacePointer &face) {
	Cluster *cl = faceClusterPairs[face];
	// Sanity check
	if (!cl) {
		return false;
	}

	for (size_t i = 0; i < 3; ++i) {
		CMeshO::FacePointer nFacePointer = face->cFFp(i);
		// Sanity checks
		if (nFacePointer == face || nFacePointer->IsD()) { continue; }

		Cluster *n_cl = faceClusterPairs[nFacePointer];
		// If the neighbor face does not belong yet to a segment
		if ( n_cl && n_cl != cl ) {
			return true;
		}
	}
	return false;
}

struct psb_help {
	psb_help(Cluster *from, Cluster *to, CMeshO::FacePointer face)
		: from_(from), to_(to), face_(face) {}

	Cluster *from_, *to_;
	CMeshO::FacePointer face_;
};

void Segmenter::processSegmentBoundaries(float eDist, float nrmDev, int inner_iterations) {
	if (cb_) {
		cb_(0, "Smoothing Segments Boundaries");
	}
	float dist = eDist;

	int outer_iter = 5;
	int mark_counter = -1;
 	for (int r = 0; r < outer_iter; ++r) {
 		int iteration = 0;
		bool finished = false;
		do {
			if (cb_) {
				cb_((100 * iteration) / inner_iterations, "Smoothing Segments Boundaries");
			}
			// initially find all edge faces
			std::vector<CMeshO::FacePointer> edgeFaces;
			edgeFaces.reserve(meshObj_->face.size() / 4);
			for (CMeshO::FaceIterator f_it = meshObj_->face.begin(); f_it != meshObj_->face.end(); ++f_it) {
				CMeshO::FacePointer facePointer = &(*f_it);
				if (isEdgeFace(facePointer)) {
					edgeFaces.push_back(facePointer);
				}
			}

			std::vector<psb_help> moveFaces;
			// process all edge faces
			for (std::vector<CMeshO::FacePointer>::iterator f_it = edgeFaces.begin(); f_it != edgeFaces.end(); ++f_it) {
				Cluster *cl = faceClusterPairs[*f_it];

				MeasureNeighborClustersFaceFunctor *mnc = new MeasureNeighborClustersFaceFunctor;
				runOperatorAtFace(*f_it, ++mark_counter, dist, mnc, dist, NULL);

				float maxArea = -FLT_MAX;
				Cluster *to = cl;

				std::map<Cluster*, float> &nClusters = mnc->getClusterData();
				for (std::map<Cluster*, float>::iterator c_it = nClusters.begin(); c_it != nClusters.end(); ++c_it) {
					if (maxArea < c_it->second) {
						maxArea = c_it->second;
						to = c_it->first;
					}
				}
				// If the best cluster is not the one the face is in and this new cluster is above 0.5% then put this face to swap list
				bool cond1 = to != cl;
				bool cond2 = (maxArea / mnc->getTotalArea()) >= 0.6f;
				bool cond3 = (nClusters[cl] / mnc->getTotalArea()) < 0.3;
				bool cond4 = metricNormalDeviation(cl->getAvgNormal(), (*f_it)->N()) < nrmDev;

				if ( cond1 && (cond2 || cond3) && cond4 ) {
					moveFaces.push_back(psb_help(cl, to, *f_it));
				}

				delete mnc;
			}	// End For
			if (!moveFaces.empty()) {
				for (size_t i = 0; i < moveFaces.size(); ++i) {
					clusterSwapFace(moveFaces[i].from_, moveFaces[i].to_, moveFaces[i].face_);
				}
			}
			// If no faces will swap segments we have reached equilibrium
			else {
				finished = true;
			}
			//std::cout << "PostProcess_VertexNeighbors Iteration " << iteration << " - Fixed " << trianglesFixed << " faces" << std::endl;
		} while (++iteration < inner_iterations);	// End Do- 
		dist = dist / 2.f;
	}	// End For (outer iterations)
	if (cb_) {
		cb_(99, "Smoothing Segments Boundaries. Finalizing");
	}
	// Now we need to search for disconnected segments.
	size_t init_clusters = clusterList_.size();
	for (size_t i = 0; i < init_clusters; ++i) {
		Cluster *cl = clusterList_[i];
		for (int j = 0; j < (int)cl->faces_.size(); ++j) {
			int mark_counter = -1;
			GatherSegmentElementsFaceFunctor *gse = new GatherSegmentElementsFaceFunctor;
			runOperatorAtFace(cl->faces_[j], ++mark_counter, FLT_MAX, gse, FLT_MAX, cl);

			std::vector<CMeshO::FacePointer> &faceList = gse->getFaceList();
			if (faceList.size() < cl->faces_.size()) {
				Cluster * to = new Cluster;
				clusterList_.push_back(to);

				for (std::vector<CMeshO::FacePointer>::iterator fh = faceList.begin(); fh != faceList.end(); ++fh) {
					clusterSwapFace(cl, to, *fh);
				}
				// Restart the check on the current cluster
				j = -1;
			}
			else {
				j = (int)cl->faces_.size() - 1;
			}
			delete gse;
		}
	}
}
